/** @file   grenade.h
 * @brief   Declaration of Grenade - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_GRENADE_H
#define H_WWW_GRENADE_H

#include "ammunition.h"

namespace WeWantWar {

/** @class  Grenade
 * @brief   Represents the grenades that can be thrown and detonated.
 * @author  Tomi Lamminsaari
 *
 * The Grenade - objects have a lifetime-property that tells how many update-
 * rounds it takes until it explodes. As it explodes, it emits some smaller
 * instances of Bullet - class to every direction.
 *
 * This class has no setter-methods so you have direct access to the
 * public members.
 *
 */
class Grenade : public Ammunition
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** The possible types of the Grenade.
   */
  enum TType {
    /** The default grenades of the game. These grenades can be detonated. */
    EDetonatable
  };
  
  
  
  ///
  /// Public members
  /// ==============
  
  /** The type of this grenade. */
  TType iType;
  
  /** Lifetime. How many updates it takes until this grenade explodes. */
  int iLifetime;
  
  /** Number of fragments we throw around us as we explode. */
  int iFragmentCount;
  
  /** How much damage each of the fragments causes. */
  int iDamage;
  
  /** A flag that tells if this Grenade has already been detonated. */
  bool iDetonated;


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	Grenade( GameObject* aOwner );


	/** Destructor
   */
	virtual ~Grenade();

private:

	/** Copy constructor.
   * @param     rO                Reference to another Grenade
   */
  Grenade( const Grenade& rO );

	/** Assignment operator
   * @param     rO                Reference to another Grenade
   * @return    Reference to us.
   */
  Grenade& operator = ( const Grenade& rO );

public:


  ///
  /// Methods
  /// =======

  /** Updates this grenade.
   * @return    <code>true</code> if the detonating time of this grenade
   *            has exceeded and grenade should be exploded.
   */
  virtual bool updateAmmunition();
  
  /** Draws this grenade on given bitmap
   * @param     aTarget           Pointer to target bitmap
   */
  virtual void redraw( BITMAP* aTarget ) const;
  
  /** Explodes this grenade.
   * @param   aBulletTable      Pointer to BulletTable where we spawn the
   *                            grenade fragments.
   */
  virtual void endingActions( BulletTable* aBulletTable );
  
  /** Detonates this Grenade. The grenade gets detonated only if the given
   * GameObject is the same who threw us.
   * @param     aOwner            Pointer to GameObject whose grenades we're
   *                              detonating. The grenade gets detonated only
   *                              if this pointer is the same who threw us or
   *                              if it's nullpointer.
   * @return    <code>true</code> if this grenade was detonated.
   */
  virtual bool detonate( GameObject* aOwner );



  ///
  /// Getter methods
  /// ==============


protected:

  ///
  /// Members
  /// =======



private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: grenade.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.1  2005-11-13 14:47:06+02  lamminsa
 * redraw() method made constant. endingActions() method added.
 *
 */
 
